/**
  *******************************************************************************
  * 
  * @file    hal_irq_cfg.c
  * @brief   config the interrupt  and read Interrupt status.
  *
  * @version v1.0
  * @date 22 Aug 2018
  * @author  AE Team
  * @note
  *
  * copyright (C) shanghai Eastsoft Mictroelectornics Co. Ltd. All rights reseverd.
  *
  @verbatim
  =================================================================================
  			##### How to use this driver #####
  =================================================================================
  [..]  The irq driver can be used as follows:
    (+) Enable/Disable specific irq.
    (+) Read specific irq status.
    (+) Read specific irq Flag.
    (+) Clear specific irq Flag.

  @endverbatim
  *********************************************************************************
  */

#include <hic.h>
#include "hal_base.h"
#include "hal_irq.h"

/**
  * @addtogroup ES7P203_HAL
  * @{
  */

/**
  * @defgroup IRQ interrupt
  * @{
  */

/**
  * @defgroup Irq_Public_Functions irq Public Functions
  * @brief    Config irq status and handle irq flag
@verbatim
=================================================================================
		##### Irq Public Functions #####
=================================================================================
  [..]  The Irq driver can be used as follows:
    (+) Config the specific irq using irq_config().
    (+) Read specific irq status using irq_get_it_status().
    (+) Read specific irq flag irq_get_flag_status().
    (+) Clear specific irq flag irq_clear_flag_status().
@endverbatim
  * @{
  */

/**
  * @brief  Enable/Disable specific irq.
  * @param  type: specific irq type, see @ref hal_irq_t.
  * @param  opt: option of specific irq, see @ref type_func_t.
  * @retval None.
  */
void irq_config(hal_irq_t type, type_func_t opt)
{
	assert_param(type <= DIVIE_IT);

	if (type > KIE_IT) {
		if (opt) {
			SET_BIT(INTE2, (1 << (type - 16)));
			return;
		}

		CLEAR_BIT(INTE2, (1 << (type - 16)));
	}
	else if (type > T21PIE_IT) {
		if (opt) {
			SET_BIT(INTE1, (1 << (type - 8)));
			return;
		}

		CLEAR_BIT(INTE1, (1 << (type - 8)));
	}
	else {
		if (opt) {
			SET_BIT(INTE0, (1 << type));
			return;
		}

		CLEAR_BIT(INTE0, (1 << type));
	}
}

/**
  * @brief  Get specific irq status.
  * @param  type: specific irq type, see @ref hal_irq_t.
  * @retval status, see @ref bit_status_t.
  */
bit_status_t irq_get_it_status(hal_irq_t type)
{
	assert_param(type <= DIVIE_IT);

	if (type > KIE_IT) {
		return READ_BIT(INTE2, (1 << (type - 16)));
	}
	else if (type > T21PIE_IT) {
		return READ_BIT(INTE1, (1 << (type - 8)));
	}
	else {
		return READ_BIT(INTE0, (1 << type));
	}

	return BIT_RESET;
}

/**
  * @brief  Get specific irq flag.
  * @param  type: specific irq type, see @ref hal_irq_t.
  * @retval status, see @ref bit_status_t.
  */
bit_status_t irq_get_flag_status(hal_irq_t type)
{
	assert_param(type <= DIVIE_IT);

	if (type > KIE_IT) {
		return READ_BIT(INTF2, (1 << (type - 16)));
	}
	else if (type > T21PIE_IT) {
		return READ_BIT(INTF1, (1 << (type - 8)));
	}
	else {
		return READ_BIT(INTF0, (1 << type));
	}

	return BIT_RESET;
}

/**
  * @brief  Clear specific irq flag.
  * @param  type: specific irq type, see @ref hal_irq_t.
  * @retval None.
  */
void irq_clear_flag_status(hal_irq_t type)
{
	assert_param(type <= DIVIE_IT);

	if (type > KIE_IT) {
		CLEAR_BIT(INTF2, (1 << (type - 16)));
	}
	else if (type > T21PIE_IT) {
		CLEAR_BIT(INTF1, (1 << (type - 8)));
	}
	else {
		CLEAR_BIT(INTF0, (1 << type));
	}

	return;
}

/**
  * @brief  Pint trigger mode config.
  * @param  mode: pint trigger mode, see @ref pint_trig_t.
  * @retval None.
  */
void pint_trig_cfg(pint_trig_t mode)
{
	assert_param((pin >= 0) && (pin <= 2));

	PEG00 = mode & 0x01;
	PEG01 = (mode >> 1) & 0x01;
	PEG02 = (mode >> 2) & 0x01;

	return;
}
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
